#include <float.h>
#include <math.h>
#include <assert.h>

#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnTransform.h>
#include <maya/MBoundingBox.h>
#include <maya/MGlobal.h>
#include <maya/MTime.h>
#include <maya/MMatrix.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MDagPathArray.h>
#include <maya/MObjectArray.h>
#include <maya/MFloatArray.h>
#include <maya/MFnMesh.h>

#include <MDt.h>
#include <MDtExt.h>

#include "rwcommon.h"

#include "querymel.h"
#include "node.h"
#include "blind.h"
#include "global.h"
#include "rpusrdat.h"

/* blindDataArray class functions */

blindDataArray::blindDataArray()
{
    name        = NULL;
    type        = invalidBlindDataType;
    format      = rpINTUSERDATA;

    intData     = NULL;
    floatData   = NULL;
    charData    = NULL;
}

blindDataArray::~blindDataArray()
{
    if (name != NULL)
    {
        free (name);
    }

    if (intData != NULL)
    {
        free (intData);
    }

    if (floatData != NULL)
    {
        free (floatData);
    }

    if (charData != NULL)
    {
        free (charData);
    }
}

bool blindDataArray::operator==(const blindDataArray &other) const
{
    if (strcmp(name, other.name) != 0)
    {
        return false;
    }

    if (type != other.type)
    {
        return false;
    }

    if (format != other.format)
    {
        return false;
    }

    return true;
}

/* blindDataSectorRemap class functions */

bool blindDataSectorRemap::operator==(const blindDataSectorRemap &other) const
{
    if (sector != other.sector)
    {
        return false;
    }
    
    return true;
}


/* blindData class functions */

blindData::blindData()
{
    numPolys    = 0;
    numVerts    = 0;
}

blindData::blindData(MObject mObject)
{
    object = mObject;

    MFnMesh mesh(object);
    
    /* Store the size of the arrays we're going to create */
    numPolys    = mesh.numPolygons();
    numVerts    = mesh.numVertices();

    /* Extra the Maya blind data */
    addMayaBlindData(MFn::kMeshVertComponent);
    addMayaBlindData(MFn::kMeshPolygonComponent);
}

blindData::~blindData()
{
    blindDataArrayListIt it;
    for (it = list.begin(); it != list.end(); it++)
    {
        delete (*it);   
    }
}

void blindData::addMayaBlindData(MFn::Type dataComponent)
{
    int             i, j, k;
    MIntArray       blindDataTemplateIDs;
    MStringArray    longNames, shortNames, formatNames;
    blindDataType   dataType;
    int             count;

    MFnMesh mesh(object);

    if (mesh.hasBlindData(dataComponent) == false)
    {
        return;
    }

    if (dataComponent == MFn::kMeshVertComponent)
    {
        dataType    = vertexBlindDataType;
        count       = numVerts;
    }
    else if(dataComponent == MFn::kMeshPolygonComponent)
    {
        dataType    = faceBlindDataType;
        count       = numPolys;
    }
    else
    {
        return;
    }

    mesh.getBlindDataTypes(dataComponent, blindDataTemplateIDs);

    /* Step through the blind data templates of the passed component type */
    for (i = 0; i < blindDataTemplateIDs.length(); i++)
    {
        char *templateName      = getBlindDataTemplateName(blindDataTemplateIDs[i]);
        char *templateNodeName  = getBlindDataTemplateNodeName(blindDataTemplateIDs[i]);
        
        if (templateName == NULL)
        {
            continue;
        }

#if MAYA_30
        mesh.getBlindDataAttrNames(blindDataTemplateIDs[i], longNames, shortNames, formatNames);
#else
        if (templateNodeName == NULL)
        {
            continue;
        }
        mesh.getBlindDataAttrNames(blindDataTemplateIDs[i], longNames, shortNames);
#endif

        /* Step through the atomic data types in this template */
        for (j = 0; j < longNames.length(); j++)
        {
            char *format = NULL;

            blindDataArray *dataArray   = new blindDataArray;
            dataArray->type             = dataType;
            dataArray->name             = (char *)malloc(sizeof(char) * strlen(templateName) +
                                                strlen(longNames[j].asChar()) + 2);
            sprintf(dataArray->name, "%s.%s", templateName, longNames[j].asChar());

#if MAYA_30
            format = strdup(formatNames[j].asChar());
#else
            format = getQueryAsString("getAttr -typ", templateNodeName, (char *)longNames[j].asChar(), true);
#endif

            if (format == NULL)
            {
                continue;
            }

            if ((strcmp(format, "int") == 0) ||(strcmp(format, "long") == 0))
            {
                MIntArray componentIDs;
                MIntArray componentValues;

                dataArray->format   = rpINTUSERDATA;
                dataArray->intData  = (int *)malloc (sizeof(int) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->intData[k] = 0;
                }

                mesh.getIntBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->intData[componentIDs[k]] = componentValues[k];
                }
                list.push_back(dataArray);
            }
            else if ((strcmp(format, "boolean") == 0) || (strcmp(format, "bool") == 0))
            {
                MIntArray componentIDs;
                MIntArray componentValues;

                dataArray->format   = rpINTUSERDATA;
                dataArray->intData  = (int *)malloc (sizeof(int) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->intData[k] = 0;
                }

                mesh.getBoolBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->intData[componentIDs[k]] = (int)componentValues[k];
                }
                list.push_back(dataArray);
            }
#ifndef MAYA_20
            else if (strcmp(format, "float") == 0)
            {
                MIntArray   componentIDs;
                MFloatArray componentValues;

                dataArray->format       = rpREALUSERDATA;
                dataArray->floatData    = (float *)malloc (sizeof(float) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->floatData[k] = 0.0f;
                }

                mesh.getFloatBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->floatData[componentIDs[k]] = componentValues[k];
                }
                list.push_back(dataArray);
            }
#endif
            else if (strcmp(format, "double") == 0)
            {
                MIntArray       componentIDs;
                MDoubleArray    componentValues;

                dataArray->format       = rpREALUSERDATA;
                dataArray->floatData    = (float *)malloc (sizeof(float) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->floatData[k] = 0.0f;
                }

                mesh.getDoubleBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->floatData[componentIDs[k]] = (float)componentValues[k];
                }
                list.push_back(dataArray);
            }
            else if (strcmp(format, "string") == 0)
            {
                MIntArray       componentIDs;
                MStringArray    componentValues;

                dataArray->format   = rpSTRINGUSERDATA;
                dataArray->charData = (char **)malloc (sizeof(char *) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->charData[k] = NULL;
                }

                mesh.getStringBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->charData[componentIDs[k]] = strdup(componentValues[k].asChar());
                }
                list.push_back(dataArray);
            }
            else if (strcmp(format, "binary") == 0)
            {
                MIntArray       componentIDs;
                MStringArray    componentValues;

                dataArray->format   = rpSTRINGUSERDATA;
                dataArray->charData = (char **)malloc (sizeof(char *) * count);

                for (k = 0; k < count; k++)
                {
                    dataArray->charData[k] = NULL;
                }

                mesh.getBinaryBlindData(dataComponent, blindDataTemplateIDs[i], longNames[j].asChar(),
                    componentIDs, componentValues);

                assert (componentIDs.length() == componentValues.length());

                for (k = 0; k < componentIDs.length(); k++)
                {
                    assert (componentIDs[k] < count);

                    dataArray->charData[componentIDs[k]] = strdup(componentValues[k].asChar());
                }
                list.push_back(dataArray);
            }
            free (format);
        }
    }
}

void blindData::remap(ReMapper &reMapper)
{
    int     i;
    int     *vertexMap;
    int     *faceMap;
    ReMapperFaceLoop    *remapFace;
    ReMapperVertex      *remapVert;

    numVerts = reMapper.GetNumVertices();
    numPolys = reMapper.GetNumFaces();

    /* Create and apply a vertex map */
    vertexMap = (int *)malloc(numVerts * sizeof(int));

    for (i = 0; i < numVerts; i++)
    {
        remapVert       = reMapper.GetVertex(i);
        vertexMap[i]    = remapVert->vertexIndex;
    }
    
    applyMap(vertexBlindDataType, vertexMap, false);
    free (vertexMap);
    
    /* Create and apply a face map */
    faceMap = (int *)malloc(numPolys * sizeof(int));
    
    for (i = 0; i < numPolys; i++)
    {
        remapFace       = reMapper.GetFace(i);
        faceMap[i]      = remapFace->faceNum;
    }

    applyMap(faceBlindDataType, faceMap, false);
    free (faceMap);
}

blindData & blindData::operator+=(blindData &b)
{
    blindDataArrayListIt    it0, it1;
    blindDataArray          *data0, *data1;
    bool                    foundMatch;
    int                     i, offset, count;
    
    /* Resize the existing blind data arrays as appropriate */
    for (it0 = list.begin(); it0 != list.end(); it0++)
    {
            data0 = *it0;
            
            if (data0->type == vertexBlindDataType)
            {
                offset  = numVerts;
                count   = b.numVerts;
            }
            else
            {
                offset  = numPolys;
                count   = b.numPolys;
            }

            switch (data0->format)
            {
                case rpINTUSERDATA:
                    data0->intData = (int *)realloc(data0->intData, sizeof(int) * (offset + count));
                
                    for (i = 0; i < count; i++)
                    {
                        data0->intData[offset + i] = 0;
                    }
                    break;

                case rpREALUSERDATA:
                    data0->floatData = (float *)realloc(data0->floatData, sizeof(float) * (offset + count));
                
                    for (i = 0; i < count; i++)
                    {
                        data0->floatData[offset + i] = 0.0f;
                    }
                    break;

                case rpSTRINGUSERDATA:
                    data0->charData = (char **)realloc(data0->charData, sizeof(char *) * (offset + count));
                
                    for (i = 0; i < count; i++)
                    {
                        data0->charData[offset + i] = NULL;
                    }
                    break;
            }
    }

    /* Go through the data arrays on b and either append each or add as a new array */
    for (it1 = b.list.begin(); it1 != b.list.end(); it1++)
    {
        data1 = *it1;

        if (data1->type == vertexBlindDataType)
        {
            offset  = numVerts;
            count   = b.numVerts;
        }
        else
        {
            offset  = numPolys;
            count   = b.numPolys;
        }

        foundMatch = false;
        
        for (it0 = list.begin(); it0 != list.end(); it0++)
        {
            data0 = *it0;

            /* Check if these these blind data arrays match */
            if (*data0 == *data1)
            {
                /* If they do then enlarge the lhs array and copy in the rhs data */
                foundMatch = true;
            
                switch (data0->format)
                {
                    case rpINTUSERDATA:
                        for (i = 0; i < count; i++)
                        {
                            data0->intData[offset + i] = data1->intData[i];
                        }
                        break;

                    case rpREALUSERDATA:
                        for (i = 0; i < count; i++)
                        {
                            data0->floatData[offset + i] = data1->floatData[i];
                        }
                        break;

                    case rpSTRINGUSERDATA:
                        for (i = 0; i < count; i++)
                        {
                            data0->charData[offset + i] = strdup(data1->charData[i]);
                        }
                        break;
                }
            }
        }

        /* We didn't find a matching data array so add a new array to the lhs */
        if (foundMatch == false)
        {
            blindDataArray *dataArray   = new blindDataArray;

            dataArray->name             = strdup(data1->name);
            dataArray->type             = data1->type;
            dataArray->format           = data1->format;

            switch (data1->format)
            {
                case rpINTUSERDATA:
                    dataArray->intData = (int *)malloc(sizeof(int) * (offset + count));
                    
                    for (i = 0; i < offset; i++)
                    {
                        dataArray->intData[i] = 0;
                    }
                    
                    for (i = 0; i < count; i++)
                    {
                        dataArray->intData[offset + i] = data1->intData[i];
                    }
                    break;

                case rpREALUSERDATA:
                    dataArray->floatData = (float *)malloc(sizeof(float) * (offset + count));
                    
                    for (i = 0; i < offset; i++)
                    {
                        dataArray->floatData[i] = 0.0f;
                    }
                    
                    for (i = 0; i < count; i++)
                    {
                        dataArray->floatData[offset + i] = data1->floatData[i];
                    }
                    break;

                case rpSTRINGUSERDATA:
                    dataArray->charData = (char **)malloc(sizeof(char *) * (offset + count));
                    
                    for (i = 0; i < offset; i++)
                    {
                        dataArray->charData[i] = NULL;
                    }
                    
                    for (i = 0; i < count; i++)
                    {
                        dataArray->charData[offset + i] = strdup(data1->charData[i]);
                    }
                    break;
            }
            list.push_back(dataArray);
        }
    }
    
    numVerts  += b.numVerts;
    numPolys  += b.numPolys;

    return *this;
}

void blindData::applyMap(blindDataType dataType, int *map, bool invert)
{
    blindDataArrayListIt    it;
    blindDataArray          *data;
    int                     *newIntData;
    float                   *newFloatData;
    char                    **newCharData;
    int                     i, count;

    switch (dataType)
    {
        case vertexBlindDataType:
            count = numVerts;
            break;
        case faceBlindDataType:
            count = numPolys;
            break;
        default:
            return;
    }

    for (it = list.begin(); it != list.end(); it++)
    {
        data = *it;

        if (data->type == dataType)
        {
            switch (data->format)
            {
                case rpINTUSERDATA:
                    newIntData = (int *)malloc(sizeof(int) * count);
            
                    for (i = 0; i < count; i++)
                    {
                        if (invert == true)
                        {
                            newIntData[map[i]]   = data->intData[i];
                        }
                        else
                        {
                            newIntData[i]   = data->intData[map[i]];
                        }
                    }
                    
                    free (data->intData);
                    data->intData = newIntData;
                    break;

                case rpREALUSERDATA:
                    newFloatData = (float *)malloc(sizeof(float) * count);
            
                    for (i = 0; i < count; i++)
                    {
                        if (invert == true)
                        {
                            newFloatData[map[i]]   = data->floatData[i];
                        }
                        else
                        {
                            newFloatData[i]   = data->floatData[map[i]];
                        }

                    }

                    free (data->floatData);
                    data->floatData = newFloatData;
                    break;

                case rpSTRINGUSERDATA:
                    newCharData = (char **)malloc(sizeof(char *) * count);
            
                    for (i = 0; i < count; i++)
                    {
                        if (invert == true)
                        {
                            newCharData[map[i]]   = strdup(data->charData[i]);
                        }
                        else
                        {
                            newCharData[i]   = strdup(data->charData[map[i]]);
                        }
                    }

                    free (data->charData);
                    data->charData = newCharData;
                    break;
            }
        }
    }
}

void blindData::addToGeometry(RpGeometry *geometry)
{
    blindDataArrayListIt    it;
    blindDataArray          *blindData;
    int                     userDataIndex;
    int                     i, elementCount, elementSize;
    RpUserDataArray         *userData;

    for (it = list.begin(); it != list.end(); it++)
    {
        blindData = *it;

        elementCount = (blindData->type == vertexBlindDataType) ? numVerts : numPolys;

        elementSize = RpUserDataGetFormatSize(blindData->format);

        userDataIndex = RpGeometryAddUserDataArray(geometry, blindData->name, blindData->format, elementCount);

        userData = RpGeometryGetUserDataArray(geometry, userDataIndex);


        if (userData != NULL)
        {
            switch (blindData->format)
            {
                case rpINTUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetInt(userData, i, blindData->intData[i]);
                    }
                    break;

                case rpREALUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetReal(userData, i, blindData->floatData[i]);
                    }
                    break;

                case rpSTRINGUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetString(userData, i, blindData->charData[i]);
                    }
                    break;
            }
        }

    }
}

RpWorldSector *blindData::addToWorldSector(RpWorldSector *worldSector, void *callBackData)
{
    blindDataArrayListIt        blindDataIt;
    blindDataArray              *blindDataArray;
    int                         i, userDataIndex, elementCount;
    RpUserDataArray             *userData;
    blindDataSectorRemapListIt  sectorIt;
    blindDataSectorRemap        *sectorRemap;
    int                         *vertexMap = NULL, *polygonMap = NULL, *currentMap;
    blindData                   *bd;

    bd = (blindData *)callBackData;

    /* Step through our list of world sector remaps and find the one for this sector */ 
    for (sectorIt = globalData->sectorMap.begin(); sectorIt != globalData->sectorMap.end(); sectorIt++)
    {
        sectorRemap = *sectorIt;

        if (sectorRemap->sector == worldSector)
        {
            vertexMap   = sectorRemap->vertexMap;
            polygonMap  = sectorRemap->polygonMap;
        }
    }

    if ((vertexMap == NULL) || (polygonMap == NULL))
    {
        return worldSector;
    }

    /* Step through all the blind data arrays we're storing */
    for (blindDataIt = bd->list.begin(); blindDataIt != bd->list.end(); blindDataIt++)
    {
        blindDataArray = *blindDataIt;

        /* Setup some variables based on the component type */
        if (blindDataArray->type == vertexBlindDataType)
        {
            elementCount    = worldSector->numVertices;
            currentMap      = vertexMap;
        }
        else
        {
            elementCount    = worldSector->numPolygons;
            currentMap      = polygonMap;
        }

        /* Add some user data to the world sector */ 
        userDataIndex   = RpWorldSectorAddUserDataArray(worldSector, blindDataArray->name, blindDataArray->format, elementCount);
        userData        = RpWorldSectorGetUserDataArray(worldSector, userDataIndex);

        /* Copy the Maya blind data into the RenderWare user data applying the sector map as we go */
        if (userData != NULL)
        {
            switch (blindDataArray->format)
            {
                case rpINTUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetInt(userData, i, blindDataArray->intData[currentMap[i]]);
                    }
                    break;

                case rpREALUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetReal(userData, i, blindDataArray->floatData[currentMap[i]]);
                    }
                    break;

                case rpSTRINGUSERDATA:
                    for (i = 0; i < elementCount; i++)
                    {
                        RpUserDataArraySetString(userData, i, blindDataArray->charData[currentMap[i]]);
                    }
                    break;
            }
        }
    }

    return worldSector;
}

/* General Functions */

void processBlineDataTemplates()
{
    int             i, j, numTemplates, numUIs;
	MStringArray    result;
    int             id;
    char            *blindDataUserName = NULL, *blindDataUserValue = NULL;

    /*
        The Maya API knowledge base recommends we do this before querying blind data.
        However, it seems to do nasty things to some scenes on Maya 2.x.
    */
#if MAYA_30
    MGlobal::executeCommand(MString("dgdirty -a"), result);
#endif
    
    /* Get a list of the blind data templates in the scene. */
    MGlobal::executeCommand( "ls -typ blindDataTemplate", result );

    numTemplates = result.length();

    /* Run through the list and store a type ID -> name map */
    for (i = 0; i < numTemplates; i++)
    {
        getQueryAsInt("getAttr", (char *)result[i].asChar(), "typeId", &id, true);
        
        getQueryAsInt("getAttr -s", (char *)result[i].asChar(), "bdUserInfo", &numUIs, true);
        
        for (j = 0; j < numUIs; j++)
        {
            char uiName[1000];

            sprintf(uiName, "bdUserInfo[%d].bdUserInfoName", j);

            blindDataUserName = getQueryAsString("getAttr", (char *)result[i].asChar(), uiName, false);
            
            if (blindDataUserName != NULL)
            {
                if (strcmp(blindDataUserName, "typeTag") == 0)
                {
                    blindDataTemplate newTemplate;

                    sprintf(uiName, "bdUserInfo[%d].bdUserInfoValue", j);
                    blindDataUserValue = getQueryAsString("getAttr", (char *)result[i].asChar(), uiName, false);

                    newTemplate.templateID          = id;
                    newTemplate.templateName        = strdup(blindDataUserValue);
                    newTemplate.templateNodeName    =  strdup((char *)result[i].asChar());

                    globalData->blindDataTemplates.push_back(newTemplate);

                    free (blindDataUserValue);
                    blindDataUserValue = NULL;
                }

                free (blindDataUserName);
                blindDataUserName = NULL;
            }
        }
    }
}

char *getBlindDataTemplateName(int templateID)
{
    char    *templateName = NULL;
    blindDataTemplateIt it;

    for (it = globalData->blindDataTemplates.begin(); it != globalData->blindDataTemplates.end(); it++)
    {
        if(it->templateID == templateID)
        {
            templateName = it->templateName;
            continue;
        }
    }

    return templateName;
}

char *getBlindDataTemplateNodeName(int templateID)
{
    char    *templateNodeName = NULL;
    blindDataTemplateIt it;

    for (it = globalData->blindDataTemplates.begin(); it != globalData->blindDataTemplates.end(); it++)
    {
        if(it->templateID == templateID)
        {
            templateNodeName = it->templateNodeName;
            continue;
        }
    }

    return templateNodeName;
}

RwBool interpVertexCB(void **pUserdataDst, void **pUserdata1, void **pUserdata2, RwReal delta)
{
    *pUserdataDst = *pUserdata1;

    return true;
}

RwBool setVertexCB(void **pUserdata, RpWorldSector *sector, RwInt32 index)
{
    blindDataSectorRemapListIt  it;
    blindDataSectorRemap        *remap;

    assert (index < sector->numVertices);

    for (it = globalData->sectorMap.begin(); it != globalData->sectorMap.end(); it++)
    {
        remap = *it;

        if (remap->sector == sector)
        {
            remap->vertexMap[index] = **((int **)pUserdata);
            return true;
        }
    }

    remap = new blindDataSectorRemap;

    remap->sector       = sector;
    remap->vertexMap    = (int *)RwMalloc(RpWorldSectorGetNumVertices(sector) * sizeof(int));
    remap->polygonMap   = (int *)RwMalloc(RpWorldSectorGetNumPolygons(sector) * sizeof(int));

    remap->vertexMap[index] = **((int **)pUserdata);

    globalData->sectorMap.push_back(remap);

    return true;
}

RwBool setPolygonCB(void **pUserdata, RpWorldSector *sector, RwInt32 index)
{
    blindDataSectorRemapListIt  it;
    blindDataSectorRemap        *remap;

    assert (index < sector->numPolygons);

    for (it = globalData->sectorMap.begin(); it != globalData->sectorMap.end(); it++)
    {
        remap = *it;

        if (remap->sector == sector)
        {
            remap->polygonMap[index] = **((int **)pUserdata);
            return true;
        }
    }

    remap = new blindDataSectorRemap;

    remap->sector       = sector;
    remap->vertexMap    = (int *)RwMalloc(RpWorldSectorGetNumVertices(sector) * sizeof(int));
    remap->polygonMap   = (int *)RwMalloc(RpWorldSectorGetNumPolygons(sector) * sizeof(int));

    remap->polygonMap[index] = **((int **)pUserdata);

    globalData->sectorMap.push_back(remap);

    return true;
}